/*
 * Valgrind trace file reader.
 */

#pragma once

#ifndef _rdr_h_
#define _rdr_h_

#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <memory>

using std::string;

#include <cstdint>
#include <cctype>

#ifdef WITH_ZLIB
#include <zlib.h>
#endif

// Value returned representing a reference.
class Reference {
public:
	enum type_t { InstrFetch, DataFetch, Store, None };
	Reference(): type(None), addr(0), size(0) { }
	Reference(type_t t, uint64_t a, int s):
		type(t), addr(a), size(s) { }
	type_t type;
	uint64_t addr;
	int size;

	bool is_fetch() { return type == InstrFetch || type == DataFetch; }
	bool is_store() { return type == Store; }
};

// This allows Reference objects to be printed with <<.  It has the 
// same purpose, but utterly different mechanism, as defining a
// toString() method in Java.
inline std::ostream & operator<<(std::ostream &strm,
				 const Reference &r) {
	strm << "["
	     << (r.type < Reference::Store ?
		 (r.type == Reference::InstrFetch ? "IF" : "DF") :
		 (r.type == Reference::Store ? "S " : "- ")) << " "
	     << std::hex << std::setfill('0') << std::setw(12) << r.addr << ":"
	     << std::dec << r.size << "]";
	return strm;
}

// Objects to fetch lines from a plain text (non-compressed) file.  First
// is an abstract interface, then one for plain I/O, then (optionally) one
// for compressed input.

// Interface for file reading objects.  C++ doesn't have interface, but
// it's easy to fake with an abstract class.
class ILineFetcher {
public:
	virtual bool eof() = 0;		// No more lines!
	virtual string getline() = 0;	// Get the next line, or ""
};

// Plain file reader.  This just reads lines from an ordinary
// text file using the standard io streams facility.
class LineFetcherPlain: public ILineFetcher {
public:
	LineFetcherPlain(string filename): in(filename) { }
	bool eof() override { return !in.good(); }
	string getline() override {
		string ret;
		if(std::getline(in, ret)) { return ret; }
		else { return ""; }
	}
private:
	std::ifstream in;
};


#ifdef WITH_ZLIB
// Gzip compressed file reader.  Created only under the WITH_ZLIB compile option.
// This just reads lines from a gzip compressed text file using the zlib.
class LineFetcherComp: public ILineFetcher {
public:
	LineFetcherComp(string filename): in(gzopen(filename.c_str(), "r")) { }
	bool eof() override { return in == NULL || gzeof(in); }
	string getline() override;
	~LineFetcherComp() { if(in) gzclose(in); }
private:
	gzFile in;	// File 
};
#endif

/*
 * Object to read trace information from the lackey file.
 */
class TraceReader {
public:
	// Construct.  Give name of file to read.  If compiled WITH_ZLIB, and
	// name ends with .gz, assumes a decompress file.  Can override this
	// behavior by sending the compressed_input parameter.
	TraceReader(string filename, bool compressed = false) {
		// The constructor creates a reader object to read the input file.
		// If compiled WITH_ZLIB, the code checks the ending of the file
		// name and optionally allocates a decompressing file reader.
		// Otherwise, no runtime if is generated, and only a plain reader
		// is created.
#ifdef WITH_ZLIB		
		if(compressed || filename.substr(filename.length()-3) == ".gz")
			m_fetcher = std::make_unique<LineFetcherComp>(filename);
		else
#endif		
			m_fetcher = std::make_unique<LineFetcherPlain>(filename);
	}
	// Read and return the next reference.  Returns a reference type None
	// if there are no more left.
	Reference read();

	// There are no more references; read() will return a type None.
	bool eof() {
		return m_pend.type == Reference::None && m_fetcher->eof();
	}
private:
	// Line fetching object.
	std::unique_ptr<ILineFetcher> m_fetcher;

	// Pending store reference.  Part of resolving an update event.
	Reference m_pend;
};

#endif
