#include <iostream>
#include <iomanip>
#include <array>
#include <chrono>
#include <random>
#include <memory>
#include <stdexcept>
#include <algorithm>

#include "AES.h"

// Generate a random salt
void AES::mksalt() {
	try {
		std::random_device devgen;
		mksalt(devgen);
	} catch(std::exception &e) {
		std::default_random_engine defgen;
		defgen.seed(std::chrono::system_clock::now().
			    time_since_epoch().count());
		mksalt(defgen);
	}
}

// Create the key and iv from existing password and salt, then initialize
// the underlying AES context.
void AES::setup()
{
	// Create the key and iv.  Assumes m_salt and m_password have
	// already been filled in.  This appears to be astoundingly difficult
	// to do well, so don't bet that I've succeeded.
	std::array<unsigned char,AES_KEYLEN+AES_BLOCKLEN> keyiv;
	keyiv.fill(0);

	int pi = 0;
	for(int i = 11; i < 3*keyiv.size(); ++i) {
		unsigned char kival = keyiv[i%keyiv.size()];
		int sft = (7*i+3) % 8;
		kival = (kival<<sft) | (kival>>8-sft);
		keyiv[i%keyiv.size()] = kival ^ 
			(unsigned char)m_password[(i+pi)%m_password.size()] ^
			m_salt[i%m_salt.size()];
		if(i % m_password.size() == 0) ++pi;
	}
	
        // Set up the library.
	AES_init_ctx_iv(&m_ctx, keyiv.data(), keyiv.data()+AES_KEYLEN);

	if(m_deflags & DUMP) {
		auto scan = keyiv.begin();
		if(m_deflags & DUMPKEY) {
			std::cerr << "Key: ";
			for(int i = AES_KEYLEN; i--; )
				std::cerr << std::setw(2) << std::hex
					  << std::setfill('0')
					  << (int)*scan++;
			std::cerr << std::endl;
		} else {
			scan += AES_KEYLEN;
		}
		if(m_deflags & DUMPIV) {
			std::cerr << " iv: ";
			for(int i = AES_BLOCKLEN; i--; )
				std::cerr << std::setw(2) << std::hex
					  << std::setfill('0')
					  << (int)*scan++;
			std::cerr << std::endl;
		}
		std::cerr << std::dec;
	}
}

// Basic crypt or decrypt on a buffer of bytes.  
void AES::crypt_append(const unsigned char *data, size_t size,
		std::vector<unsigned char> &result,
		bool complete /* = false */)
{
	if(m_first) {
		if(m_encrypt) {
			// Generate a random salt, and add it to
			// the result so it can be read back in on
			// decrypt.
			mksalt();
			if(!(m_deflags & NOSALT)) {
				result.insert(result.end(),
					      m_salt.begin(), m_salt.end());
			}
		} else {
			// Decrypting is harder, since we have to read the
			// salt from the input, and the client may not
			// send it all at once.

			// Copy input to salt, up to its size.
			long saltamt = std::min<long>
				(size, m_salt.end() - m_saltscan);
			m_saltscan = std::copy(data, data+saltamt, m_saltscan);

			// If salt is not full, wait for more data.  Will
			// return no bytes since there is no actual data yet.
			// Note that m_first is still set, so we will enter
			// here again on the next call.
			if(m_saltscan < m_salt.end()) return;

			// Remove the data from the buffer.
			data += saltamt;
			size -= saltamt;
		}

		// We're go.  Set the parameters from the the seed and pwd.
		setup();
		m_first = false;
	}

	// Okay.  Now we can do something with the data.  We'll work in
	// blocks since it's easier.  The underlying code will do this
	// anyway.

	// Build our blocks here.
	std::array<unsigned char, AES_BLOCKLEN> sendbuf;
	auto sat = sendbuf.begin();
	while(true) {
		// Comsume any pending.
		sat = sendbuf.begin();
		if(m_pendscan > m_pend.begin()) {
			sat = std::copy(m_pend.begin(), m_pendscan, sat);
			m_pendscan = m_pend.begin();
		}

		// Fill up sendbuf with data from the user, up to either the
		// buffer capacity or the amount of available data.
		long amt = std::min<long>(size, sendbuf.end() - sat);
		sat = std::copy(data, data+amt, sat);
		data += amt;
		size -= amt;
		
		// Didn't fill the block.  Don't translate.
		if(sat < sendbuf.end()) break;

		// Translate the buffer.
		if(m_encrypt)
			AES_CBC_encrypt_buffer(&m_ctx,
					       sendbuf.data(), sendbuf.size());
		else {
			AES_CBC_decrypt_buffer(&m_ctx,
					       sendbuf.data(), sendbuf.size());
			// Returning the last decrypt block requires some
			// special processing, which we'll do later.
			if(complete && size == 0) break;
		}

		// Add converted buffer data to result.
		result.insert(result.end(), sendbuf.begin(), sendbuf.end());
	}

	// If there's just some extra, buffer it.
	if(!complete) {
		// Pend the balance.
		m_pendscan = std::copy(sendbuf.begin(), sat, m_pend.begin());
		return;
	}

	// Finish off
	if(m_encrypt) {
		// Pad and complete.  Note that if we came out even
		// exactly, we're going to add a full block of padding
		// so we can always identify the end on decrypt.
		int extra = sendbuf.end() - sat;
		while(sat < sendbuf.end()) *sat++ = extra;
		AES_CBC_encrypt_buffer(&m_ctx,
				       sendbuf.data(),sendbuf.size());
		result.insert(result.end(),
			      sendbuf.begin(), sendbuf.end());
	} else {
		// Decrypt should always be a multiple of buffer size.
		if(sat != sendbuf.end()) {
			throw std::runtime_error
				("AES Decrypt stream length not "
				 "multiple of buffer size.");
		}
		
		// Check the padding.
		auto scan = sendbuf.end();
		unsigned char padsize = scan[-1];
		for(int i = 0; i < padsize; ++i)
			if(*--scan != padsize)
				throw std::runtime_error("AES Decrypt stream "
							 "padding error");
		
		// Add to the result.
		result.insert(result.end(), sendbuf.begin(), scan);
	}
}
