/*
 * An object to encrypt or decrypt data using AES in CBC mode.  The underlying
 * engine is Tiny AES, https://github.com/kokke/tiny-AES-c.
 *
 * The constructor accepts a string password and a boolean encrypt flag: send
 * true to encrypt the stream, false to decrypt it.
 *
 * A block of data is encrypted or decrypted with crypt.  The translated data
 * is returned in an argument vector.  The last parameter to crypt identifies
 * the last block of data.  It must be false for every block before the last,
 * and true for the last.
 *
 *     unsigned char arr[1024];
 *     int howmuch = how_much_of_arr_is_valid();
 *     std::vector<unsigned char> in;
 *     std::vector<unsigned char> out;
 *     std::string strin;
 *
 *     AES aes("password", true);
 *     aes.crypt(arr, howmuch, out);
 *     aes.crypt(in, out);
 *     aes.crypt(strin, out);
 *
 * But the last time
 *     aes.crypt(arr, howmuch, out, true);
 *     ..etc..
 *
 * Additional interface has crypt_append, which adds data to the result
 * rather than replacing it.
 *
 * Input data blocks may be of any positive size.  Return blocks may be any
 * size, including zero.  Encryption adds an 8-byte salt to the front of the
 * encrypted stream, and necessary padding to the end.  Decryption expects to
 * receive a stream that came from the encryption function of this class.  It
 * will return the same bytes same order which were originally send to
 * encryption, and no others, but very likely in different-sized buffers.
 * Decryption checks that the padding is correct, and will throw
 * std::runtime_error if it is not.  This usually means a bad password or some
 * corruption of the input stream.  Most such errors will produce an exception,
 * but there is no guarantee.
 */

#pragma once
#ifndef _AES_H_
#define _AES_H_

#include <string>
#include <vector>
#include <random>
#include <array>
#include <stdexcept>

extern "C" {
#include "tiny_aes.h"	
}

class AES {
public:
	// Construct with a password used to generate the key and IV.
	AES(std::string password, bool enc):
		m_first(true), m_pendscan(m_pend.begin()), m_deflags(0),
		m_saltscan(m_salt.begin()), m_password(password),
		m_encrypt(enc) { }

	// Basic crypt or decrypt on a buffer of bytes.  Puts result into
	// argument vector.  False indicates the end of the encryption.
	void crypt_append(const unsigned char *data, size_t size,
		   std::vector<unsigned char> &result, bool complete = false);
	void crypt(const unsigned char *data, size_t size,
		   std::vector<unsigned char> &result, bool complete = false)
	{
		result.clear();
		crypt_append(data,size,result,complete);
	}

	// This will work with a C++ container with byte content, and
	// with a .data() method.  That's vectors, arrays and strings mostly.
	// Remember, though, that the size of std::array is its physical size,
	// so don't use one of those unless your array is actually full.
	template<typename ContainerType>
	void crypt_append(const ContainerType &d,
			  std::vector<unsigned char> &result,
			  bool complete = false)
	{
		char failmaker[2 - (int)sizeof(typename ContainerType::value_type)];
		crypt_append(reinterpret_cast<const unsigned char *>(d.data()),
			     d.size(), result, complete);
	}
	template<typename ContainerType>
	void crypt(const ContainerType &d,
			  std::vector<unsigned char> &result,
			  bool complete = false)
	{
		result.clear();
		crypt_append(d,result,complete);
	}

	// This one is supposed to accomodate std::array if you also need to
	// send a size.
	template<typename T, size_t N>
	void crypt_append(const std::array<T,N> &a, size_t size,
			  std::vector<unsigned char> &result,
			  bool complete = false)
	{
		char failmaker[2 - (int)sizeof(T)];
		if(size > N)
			throw std::range_error
				("AES::crypt: size " + std::to_string(size) +
				 " exceeds container " + std::to_string(N));
		crypt_append(reinterpret_cast<const unsigned char *>(a.data()),
			     size, result, complete);
	}
	template<typename T, size_t N>
	void crypt(const std::array<T,N> &a, size_t size,
			  std::vector<unsigned char> &result,
			  bool complete = false)
	{
		result.clear();
		crypt_append(a,size,result,complete);
	}
private:
	// Generate a random salt using a specified generator.
	template<typename gen_t>
	void mksalt(gen_t &generator) {
		std::uniform_int_distribution<int> dist(0,255);
		for(auto &x: m_salt) x = dist(generator);
	}

	// Generate a random salt
	void mksalt();
	
	// Call the init based on m_seed and the m_user.  Generates the
	// key and iv from those values.
	void setup();
		
	// State from the tiny_aes library.
	struct AES_ctx m_ctx;

	// Crypt/decrypt
	bool m_encrypt;

	// Remember the password
	std::string m_password;

	// The salt.
	static const int SALTSIZE = 8;
	std::array<unsigned char, SALTSIZE> m_salt;
	std::array<unsigned char, SALTSIZE>::iterator m_saltscan;
	
	// Pending bytes.  This is a part of a block for when an input is
	// not divisible by the block size.
	std::array<unsigned char, AES_BLOCKLEN> m_pend;
	std::array<unsigned char, AES_BLOCKLEN>::iterator m_pendscan;

	// This is the first operation, so handle the iv.
	bool m_first;

	// Testing flags.
	unsigned int m_deflags;
public:
	static const unsigned int NOSALT = 0x0001;	// Don't output salt.
	static const unsigned int DUMPKEY = 0x0002;
	static const unsigned int DUMPIV = 0x0004;
	static const unsigned int DUMP = (DUMPKEY|DUMPIV);
	void debug(unsigned int flags) { m_deflags |= flags; }
};

#endif
