/*
 * Test program for the AES class.
 *     aestest [ -d ] [ -s ] [ -b bufsize ] en|de password filename
 * Encrypts (en) or decrypts (de) the named file.  Reads or writes plaintext
 * on standard streams.  Normally reads 1024 bytes at a time, but this can
 * be changed (to a smaller value) with the -b option.  The other flags control
 * debug options for the AES object: -d dumps the keying info, and -s omits
 * the salt prefix in encrypted output.
 */
#include <cstring>
#include <array>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <string>
#include <vector>
#include <algorithm>

#include <cstdlib>

#include "AES.h"

using namespace std;

void perform(AES &aes, istream &in, ostream &out, int bufsize);

void dump(unsigned char *data, int size);

int main(int argc, char **argv)
{
	bool dump = false;
	int bufsize = -1;
	bool nosalt = false;

	string pname = argv[0];
	--argc;
	++argv;
	
	while(argc > 0 && **argv == '-') {
		int inc = 1;
		if(string(*argv) == "-d") {
			dump = true;
		} else if(string(*argv) == "-s") {
			nosalt = true;
		} else if(string(*argv) == "-b" && argc > 1) {
			bufsize = atoi(argv[1]);
			inc = 2;
		}

		argc -= inc;
		argv += inc;
	}
	
	if(argc != 3 || (argv[0] != string("en") && argv[0] != string("de"))) {
		cerr << pname << " [ -d ] [ -s ] [ -b size ] "
		     << "en|de password filename" << endl;
		cerr << "  Named file is the encrypted one, output for en and "
		     << "input for de" << endl;
		cerr << "  -d dumps keying; -s omits salt; -b set buffer size"
		     << endl;
		exit(2);
	}

	bool encrypt = (argv[0] == string("en"));

	fstream file;
	file.open(argv[2],
		  (encrypt?ios_base::out:ios_base::in)|ios_base::binary);
	if(!file) {
		cerr << "Unable to open " << argv[2] << endl;
		exit(3);
	}

	AES docrypt(argv[1], encrypt);
	if(dump) docrypt.debug(AES::DUMP);
	if(nosalt) docrypt.debug(AES::NOSALT);

	if(encrypt)
		perform(docrypt, cin, file, bufsize);
	else {
		perform(docrypt, file, cout, bufsize);
	}
}

void perform(AES &aes, istream &in, ostream &out, int bufsize)
{
	array<unsigned char, 1024> inbuf;
	vector<unsigned char> outbuf;

	// Fake buffer size for testing.
	if(bufsize > 0) {
		bufsize = min<int>(inbuf.size(), bufsize);
		srand(time(NULL));
	} else {
		bufsize = inbuf.size();
	}
	
	while(in.good()) {
		in.read(reinterpret_cast<char *>(inbuf.data()), bufsize);
		int nread = in.gcount();
		in.peek();  // Lights EOF flag if there's no more to read.
		aes.crypt(inbuf.data(), nread, outbuf, !in.good());
		out.write(reinterpret_cast<char *>(outbuf.data()),
			  outbuf.size());
	}
}
		
void dump(unsigned char *data, int size)
{
	cout << "\n==[";
	while(size--) {
		cout << setw(2) << setfill('0') << hex <<
			((unsigned int)*data++ & 0x0ff) << dec;
	}
	cout << "]==\n" << endl;
}
